/**
 * CONTACT FORM WIDGET
 * ===================
 * Customizable contact form with validation and email integration.
 *
 * Config options:
 * - fields: Array of field configs
 * - submitUrl: URL to POST form data
 * - submitMethod: 'email' | 'webhook' | 'api'
 * - emailTo: Email address for submissions
 * - successMessage: Message after successful submit
 * - successAction: 'message' | 'redirect' | 'hide'
 * - redirectUrl: URL for redirect action
 * - buttonText: Submit button text
 * - buttonColor: Button background color
 * - theme: 'light' | 'dark' | 'minimal'
 * - layout: 'stacked' | 'inline' | 'two-column'
 * - showLabels: true/false
 * - floatingLabels: true/false
 * - borderRadius: CSS value
 */

window.FaresWidgets = window.FaresWidgets || {};

window.FaresWidgets.contact_form = {
    defaultConfig: {
        fields: [
            { name: 'name', type: 'text', label: 'Full Name', required: true, placeholder: 'John Smith' },
            { name: 'email', type: 'email', label: 'Email', required: true, placeholder: 'john@example.com' },
            { name: 'phone', type: 'tel', label: 'Phone', required: false, placeholder: '(555) 123-4567' },
            { name: 'message', type: 'textarea', label: 'Message', required: true, placeholder: 'How can we help you?' }
        ],
        submitUrl: '',
        submitMethod: 'webhook',
        emailTo: '',
        successMessage: 'Thank you! We\'ll be in touch soon.',
        successAction: 'message',
        redirectUrl: '',
        buttonText: 'Send Message',
        buttonColor: '#2563eb',
        theme: 'light',
        layout: 'stacked',
        showLabels: true,
        floatingLabels: false,
        borderRadius: '8px',
        recaptcha: false,
        recaptchaSiteKey: ''
    },

    init: function(container, config) {
        this.container = container;
        this.config = { ...this.defaultConfig, ...config };
        this.render();
        this.attachEvents();
    },

    render: function() {
        const {
            fields, buttonText, buttonColor, theme,
            layout, showLabels, floatingLabels, borderRadius
        } = this.config;

        this.addStyles();

        const formId = 'fares-form-' + Math.random().toString(36).substr(2, 9);

        this.container.innerHTML = `
            <form class="fares-form fares-form-${theme} fares-form-${layout} ${floatingLabels ? 'floating' : ''}"
                  id="${formId}" style="--radius:${borderRadius};">

                <div class="fares-form-fields">
                    ${fields.map(field => this.renderField(field)).join('')}
                </div>

                <div class="fares-form-submit">
                    <button type="submit" class="fares-form-btn" style="background:${buttonColor};">
                        <span class="fares-form-btn-text">${buttonText}</span>
                        <span class="fares-form-btn-loading" style="display:none;">
                            <svg class="fares-spinner" viewBox="0 0 24 24">
                                <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="3" fill="none" stroke-dasharray="30 70"/>
                            </svg>
                            Sending...
                        </span>
                    </button>
                </div>

                <div class="fares-form-success" style="display:none;">
                    <svg viewBox="0 0 24 24" fill="currentColor" width="48" height="48">
                        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
                    </svg>
                    <p>${this.config.successMessage}</p>
                </div>

                <div class="fares-form-error" style="display:none;">
                    <p></p>
                </div>
            </form>
        `;

        this.form = this.container.querySelector('form');
        this.btnText = this.container.querySelector('.fares-form-btn-text');
        this.btnLoading = this.container.querySelector('.fares-form-btn-loading');
        this.successEl = this.container.querySelector('.fares-form-success');
        this.errorEl = this.container.querySelector('.fares-form-error');
    },

    renderField: function(field) {
        const { showLabels, floatingLabels } = this.config;
        const required = field.required ? 'required' : '';
        const labelHtml = showLabels ? `<label for="${field.name}">${field.label}${field.required ? ' *' : ''}</label>` : '';

        let inputHtml = '';

        switch (field.type) {
            case 'textarea':
                inputHtml = `
                    <textarea name="${field.name}" id="${field.name}"
                              placeholder="${field.placeholder || ''}"
                              rows="${field.rows || 4}" ${required}></textarea>
                `;
                break;

            case 'select':
                inputHtml = `
                    <select name="${field.name}" id="${field.name}" ${required}>
                        <option value="">${field.placeholder || 'Select...'}</option>
                        ${(field.options || []).map(opt =>
                            `<option value="${opt.value || opt}">${opt.label || opt}</option>`
                        ).join('')}
                    </select>
                `;
                break;

            case 'checkbox':
                return `
                    <div class="fares-form-field fares-form-checkbox">
                        <label>
                            <input type="checkbox" name="${field.name}" ${required}>
                            <span>${field.label}</span>
                        </label>
                    </div>
                `;

            case 'radio':
                return `
                    <div class="fares-form-field fares-form-radio">
                        ${showLabels ? `<span class="fares-form-radio-label">${field.label}</span>` : ''}
                        ${(field.options || []).map(opt => `
                            <label>
                                <input type="radio" name="${field.name}" value="${opt.value || opt}" ${required}>
                                <span>${opt.label || opt}</span>
                            </label>
                        `).join('')}
                    </div>
                `;

            default: // text, email, tel, number, etc.
                inputHtml = `
                    <input type="${field.type}" name="${field.name}" id="${field.name}"
                           placeholder="${field.placeholder || ''}" ${required}
                           ${field.pattern ? `pattern="${field.pattern}"` : ''}>
                `;
        }

        return `
            <div class="fares-form-field ${field.width === 'half' ? 'half' : ''}">
                ${floatingLabels ? inputHtml + labelHtml : labelHtml + inputHtml}
            </div>
        `;
    },

    addStyles: function() {
        if (document.getElementById('fares-form-styles')) return;

        const styles = document.createElement('style');
        styles.id = 'fares-form-styles';
        styles.textContent = `
            .fares-form {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                --radius: 8px;
            }
            .fares-form-light {
                --bg: #fff;
                --input-bg: #f9fafb;
                --border: #e5e7eb;
                --text: #111827;
                --text-muted: #6b7280;
            }
            .fares-form-dark {
                --bg: #1f2937;
                --input-bg: #374151;
                --border: #4b5563;
                --text: #f9fafb;
                --text-muted: #9ca3af;
            }
            .fares-form-minimal {
                --bg: transparent;
                --input-bg: transparent;
                --border: #d1d5db;
                --text: inherit;
                --text-muted: #6b7280;
            }
            .fares-form-fields {
                display: flex;
                flex-direction: column;
                gap: 16px;
            }
            .fares-form-two-column .fares-form-fields {
                display: grid;
                grid-template-columns: repeat(2, 1fr);
                gap: 16px;
            }
            .fares-form-two-column .fares-form-field.half {
                grid-column: span 1;
            }
            .fares-form-two-column .fares-form-field:not(.half) {
                grid-column: span 2;
            }
            .fares-form-field label {
                display: block;
                font-size: 14px;
                font-weight: 500;
                color: var(--text);
                margin-bottom: 6px;
            }
            .fares-form-field input,
            .fares-form-field textarea,
            .fares-form-field select {
                width: 100%;
                padding: 12px 14px;
                font-size: 15px;
                border: 1px solid var(--border);
                border-radius: var(--radius);
                background: var(--input-bg);
                color: var(--text);
                transition: border-color 0.2s, box-shadow 0.2s;
                box-sizing: border-box;
            }
            .fares-form-field input:focus,
            .fares-form-field textarea:focus,
            .fares-form-field select:focus {
                outline: none;
                border-color: #2563eb;
                box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
            }
            .fares-form-field input::placeholder,
            .fares-form-field textarea::placeholder {
                color: var(--text-muted);
            }
            .fares-form-field textarea {
                resize: vertical;
                min-height: 100px;
            }
            .fares-form.floating .fares-form-field {
                position: relative;
            }
            .fares-form.floating .fares-form-field label {
                position: absolute;
                left: 14px;
                top: 50%;
                transform: translateY(-50%);
                pointer-events: none;
                transition: 0.2s;
                background: var(--bg);
                padding: 0 4px;
                margin: 0;
            }
            .fares-form.floating .fares-form-field input:focus + label,
            .fares-form.floating .fares-form-field input:not(:placeholder-shown) + label {
                top: 0;
                font-size: 12px;
                color: #2563eb;
            }
            .fares-form-checkbox label,
            .fares-form-radio label {
                display: flex;
                align-items: center;
                gap: 8px;
                font-weight: normal;
                cursor: pointer;
            }
            .fares-form-checkbox input,
            .fares-form-radio input {
                width: auto;
            }
            .fares-form-submit {
                margin-top: 20px;
            }
            .fares-form-btn {
                width: 100%;
                padding: 14px 24px;
                font-size: 16px;
                font-weight: 600;
                color: #fff;
                border: none;
                border-radius: var(--radius);
                cursor: pointer;
                transition: opacity 0.2s, transform 0.2s;
                display: flex;
                align-items: center;
                justify-content: center;
                gap: 8px;
            }
            .fares-form-btn:hover {
                opacity: 0.9;
            }
            .fares-form-btn:active {
                transform: scale(0.98);
            }
            .fares-form-btn:disabled {
                opacity: 0.6;
                cursor: not-allowed;
            }
            .fares-spinner {
                width: 20px;
                height: 20px;
                animation: fares-spin 1s linear infinite;
            }
            @keyframes fares-spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
            .fares-form-success {
                text-align: center;
                padding: 40px 20px;
                color: #059669;
            }
            .fares-form-success svg {
                margin-bottom: 16px;
            }
            .fares-form-success p {
                font-size: 18px;
                margin: 0;
            }
            .fares-form-error {
                padding: 12px 16px;
                background: #fef2f2;
                border: 1px solid #fecaca;
                border-radius: var(--radius);
                color: #dc2626;
                margin-top: 16px;
            }
            .fares-form-error p {
                margin: 0;
                font-size: 14px;
            }
            @media (max-width: 600px) {
                .fares-form-two-column .fares-form-fields {
                    grid-template-columns: 1fr;
                }
                .fares-form-two-column .fares-form-field {
                    grid-column: span 1 !important;
                }
            }
        `;
        document.head.appendChild(styles);
    },

    attachEvents: function() {
        this.form.addEventListener('submit', (e) => this.handleSubmit(e));
    },

    async handleSubmit(e) {
        e.preventDefault();

        const { submitUrl, submitMethod, emailTo, successAction, redirectUrl } = this.config;

        // Show loading state
        this.btnText.style.display = 'none';
        this.btnLoading.style.display = 'flex';
        this.form.querySelector('button').disabled = true;

        // Gather form data
        const formData = new FormData(this.form);
        const data = Object.fromEntries(formData.entries());

        try {
            let response;

            switch (submitMethod) {
                case 'email':
                    // Use mailto (client-side)
                    const subject = `Contact Form: ${data.name || 'New Submission'}`;
                    const body = Object.entries(data).map(([k, v]) => `${k}: ${v}`).join('\n');
                    window.location.href = `mailto:${emailTo}?subject=${encodeURIComponent(subject)}&body=${encodeURIComponent(body)}`;
                    break;

                case 'api':
                case 'webhook':
                default:
                    response = await fetch(submitUrl || 'https://calidream-calculators.web.app/api/contact', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(data)
                    });

                    if (!response.ok) throw new Error('Failed to submit');
            }

            // Handle success
            switch (successAction) {
                case 'redirect':
                    window.location.href = redirectUrl;
                    break;
                case 'hide':
                    this.container.style.display = 'none';
                    break;
                default: // message
                    this.form.querySelector('.fares-form-fields').style.display = 'none';
                    this.form.querySelector('.fares-form-submit').style.display = 'none';
                    this.successEl.style.display = 'block';
            }

        } catch (error) {
            this.errorEl.querySelector('p').textContent = 'Something went wrong. Please try again.';
            this.errorEl.style.display = 'block';
        } finally {
            this.btnText.style.display = 'inline';
            this.btnLoading.style.display = 'none';
            this.form.querySelector('button').disabled = false;
        }
    }
};
