/**
 * SOCIAL PROOF POPUP WIDGET
 * =========================
 * "X just purchased..." style FOMO notifications.
 * Proven to increase conversions by showing real-time activity.
 *
 * Config options:
 * - notifications: Array of {name, location, action, product, time, image}
 * - autoGenerate: true/false (generate fake notifications)
 * - position: 'bottom-left' | 'bottom-right' | 'top-left' | 'top-right'
 * - delay: ms before first popup
 * - interval: ms between popups
 * - duration: ms popup stays visible
 * - maxPerSession: max popups to show per session
 * - theme: 'light' | 'dark' | 'minimal'
 * - showImage: true/false
 * - showTime: true/false
 * - sound: true/false
 */

window.FaresWidgets = window.FaresWidgets || {};

window.FaresWidgets.social_proof_popup = {
    defaultConfig: {
        notifications: [
            { name: "Michael", location: "San Diego, CA", action: "just requested", product: "a free estimate", time: "2 minutes ago" },
            { name: "Sarah", location: "La Jolla, CA", action: "just booked", product: "a consultation", time: "5 minutes ago" },
            { name: "David", location: "Carlsbad, CA", action: "just signed up", product: "for our newsletter", time: "8 minutes ago" },
            { name: "Jennifer", location: "Encinitas, CA", action: "just purchased", product: "the Starter Plan", time: "12 minutes ago" },
            { name: "Robert", location: "Del Mar, CA", action: "just downloaded", product: "our free guide", time: "15 minutes ago" }
        ],
        autoGenerate: false,
        position: 'bottom-left',
        delay: 3000,
        interval: 8000,
        duration: 5000,
        maxPerSession: 10,
        theme: 'light',
        showImage: true,
        showTime: true,
        sound: false,
        accentColor: '#10b981',
        borderRadius: '12px',
        animation: 'slide', // 'slide' | 'fade' | 'bounce'
        showCloseButton: true
    },

    init: function(container, config) {
        this.config = { ...this.defaultConfig, ...config };
        this.currentIndex = 0;
        this.shown = 0;
        this.render();
        this.startLoop();
    },

    render: function() {
        this.addStyles();

        // Create popup container
        this.popup = document.createElement('div');
        this.popup.className = `fares-sp-popup fares-sp-${this.config.position} fares-sp-${this.config.theme}`;
        this.popup.style.display = 'none';
        document.body.appendChild(this.popup);
    },

    addStyles: function() {
        if (document.getElementById('fares-sp-styles')) return;

        const { accentColor, borderRadius, animation } = this.config;

        const styles = document.createElement('style');
        styles.id = 'fares-sp-styles';
        styles.textContent = `
            .fares-sp-popup {
                position: fixed;
                z-index: 99999;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                max-width: 340px;
                --accent: ${accentColor};
                --radius: ${borderRadius};
            }
            .fares-sp-bottom-left { bottom: 20px; left: 20px; }
            .fares-sp-bottom-right { bottom: 20px; right: 20px; }
            .fares-sp-top-left { top: 20px; left: 20px; }
            .fares-sp-top-right { top: 20px; right: 20px; }

            .fares-sp-light {
                background: #fff;
                color: #111827;
                box-shadow: 0 10px 40px rgba(0,0,0,0.15);
            }
            .fares-sp-dark {
                background: #1f2937;
                color: #f9fafb;
                box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            }
            .fares-sp-minimal {
                background: rgba(255,255,255,0.95);
                backdrop-filter: blur(10px);
                color: #111827;
                box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            }

            .fares-sp-popup {
                border-radius: var(--radius);
                padding: 16px;
                display: flex;
                align-items: center;
                gap: 14px;
                transform: translateX(-120%);
                opacity: 0;
                transition: transform 0.4s ease, opacity 0.4s ease;
            }
            .fares-sp-popup.visible {
                transform: translateX(0);
                opacity: 1;
            }
            .fares-sp-bottom-right.fares-sp-popup,
            .fares-sp-top-right.fares-sp-popup {
                transform: translateX(120%);
            }
            .fares-sp-bottom-right.visible,
            .fares-sp-top-right.visible {
                transform: translateX(0);
            }

            .fares-sp-image {
                width: 48px;
                height: 48px;
                border-radius: 50%;
                background: var(--accent);
                display: flex;
                align-items: center;
                justify-content: center;
                color: #fff;
                font-weight: 600;
                font-size: 18px;
                flex-shrink: 0;
            }
            .fares-sp-image img {
                width: 100%;
                height: 100%;
                border-radius: 50%;
                object-fit: cover;
            }

            .fares-sp-content {
                flex: 1;
                min-width: 0;
            }
            .fares-sp-text {
                font-size: 14px;
                line-height: 1.4;
            }
            .fares-sp-name {
                font-weight: 600;
            }
            .fares-sp-action {
                color: inherit;
            }
            .fares-sp-product {
                font-weight: 600;
                color: var(--accent);
            }
            .fares-sp-meta {
                font-size: 12px;
                color: #6b7280;
                margin-top: 4px;
                display: flex;
                align-items: center;
                gap: 8px;
            }
            .fares-sp-dark .fares-sp-meta {
                color: #9ca3af;
            }
            .fares-sp-verified {
                color: var(--accent);
            }

            .fares-sp-close {
                position: absolute;
                top: 8px;
                right: 8px;
                width: 20px;
                height: 20px;
                border: none;
                background: none;
                color: #9ca3af;
                cursor: pointer;
                font-size: 16px;
                display: flex;
                align-items: center;
                justify-content: center;
                border-radius: 50%;
                transition: all 0.2s;
            }
            .fares-sp-close:hover {
                background: #f3f4f6;
                color: #374151;
            }
            .fares-sp-dark .fares-sp-close:hover {
                background: #374151;
                color: #f9fafb;
            }

            @media (max-width: 480px) {
                .fares-sp-popup {
                    left: 10px;
                    right: 10px;
                    max-width: none;
                }
            }
        `;
        document.head.appendChild(styles);
    },

    startLoop: function() {
        // Initial delay
        setTimeout(() => {
            this.showNext();
        }, this.config.delay);
    },

    showNext: function() {
        if (this.shown >= this.config.maxPerSession) return;

        const { notifications, autoGenerate } = this.config;
        let notification;

        if (autoGenerate) {
            notification = this.generateNotification();
        } else {
            notification = notifications[this.currentIndex % notifications.length];
            this.currentIndex++;
        }

        this.showPopup(notification);
        this.shown++;

        // Schedule next
        setTimeout(() => {
            this.showNext();
        }, this.config.interval);
    },

    generateNotification: function() {
        const names = ["Michael", "Sarah", "David", "Jennifer", "Robert", "Emily", "James", "Lisa", "Chris", "Amanda"];
        const locations = ["San Diego", "La Jolla", "Carlsbad", "Encinitas", "Del Mar", "Oceanside", "Escondido"];
        const actions = ["just requested", "just signed up for", "just downloaded", "just purchased"];
        const products = ["a free estimate", "a consultation", "our newsletter", "the Pro Plan"];
        const times = ["just now", "1 minute ago", "2 minutes ago", "5 minutes ago"];

        return {
            name: names[Math.floor(Math.random() * names.length)],
            location: locations[Math.floor(Math.random() * locations.length)] + ", CA",
            action: actions[Math.floor(Math.random() * actions.length)],
            product: products[Math.floor(Math.random() * products.length)],
            time: times[Math.floor(Math.random() * times.length)]
        };
    },

    showPopup: function(notification) {
        const { showImage, showTime, showCloseButton, duration } = this.config;

        this.popup.innerHTML = `
            ${showCloseButton ? '<button class="fares-sp-close">×</button>' : ''}
            ${showImage ? `
                <div class="fares-sp-image">
                    ${notification.image
                        ? `<img src="${notification.image}" alt="${notification.name}">`
                        : notification.name.charAt(0)
                    }
                </div>
            ` : ''}
            <div class="fares-sp-content">
                <div class="fares-sp-text">
                    <span class="fares-sp-name">${notification.name}</span>
                    from ${notification.location}
                    <span class="fares-sp-action">${notification.action}</span>
                    <span class="fares-sp-product">${notification.product}</span>
                </div>
                ${showTime ? `
                    <div class="fares-sp-meta">
                        <span>${notification.time}</span>
                        <span class="fares-sp-verified">✓ Verified</span>
                    </div>
                ` : ''}
            </div>
        `;

        // Show
        this.popup.style.display = 'flex';
        requestAnimationFrame(() => {
            this.popup.classList.add('visible');
        });

        // Play sound
        if (this.config.sound) {
            this.playSound();
        }

        // Close button
        const closeBtn = this.popup.querySelector('.fares-sp-close');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => this.hidePopup());
        }

        // Auto hide
        setTimeout(() => this.hidePopup(), duration);
    },

    hidePopup: function() {
        this.popup.classList.remove('visible');
        setTimeout(() => {
            this.popup.style.display = 'none';
        }, 400);
    },

    playSound: function() {
        // Simple notification sound
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();

        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);

        oscillator.frequency.value = 800;
        oscillator.type = 'sine';
        gainNode.gain.value = 0.1;

        oscillator.start();
        oscillator.stop(audioContext.currentTime + 0.1);
    }
};
